"""
--------------------------------------------------------------------------------
gps_on_bbb.py
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
Copyright (C) 2015 nimbelink.com, MIT License

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the "Software"), to deal in
the Software without restriction, including without limitation the rights to
use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
the Software, and to permit persons to whom the Software isfurnished to do so,
subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
-------------
PREREQUESITES
-------------
This code uses the following Python libarires that must be installed:
Requests:   http://www.python-requests.org/en/latest/
pySerial:   http://pyserial.sourceforge.net/index.html

-----------
DESCRIPTION
-----------
This sends GPS information from a Skywire DE910 or LE910 to dweet.io over PPP on
a BeagleBone Black.

Please consult the application note for package install instrutions.
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
----------
CHANGE LOG
----------
v1 (1/2016): Initial
v2 (3/15/16): Fixed issue where you would need to power cycle the BeagleBone Black in
    order to run the Python script again
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
"""

import requests
import serial
from time import sleep

#   Set the modem you are using to True
DE910 = True
LE910 = False

#   URL for sending to dweet.io is assigned by getMEID() below
URL = ""

#   Setup serial connection
"""
NOTE:   For the DE910 or LE910, you can use either the UART port (/dev/ttyO4) or the
        secondary USB port (/dev/ttyUSB2). This code has been tested with both.
        Make sure only one is uncommented.
"""
#   Serial using the UART port
skywire = serial.Serial("/dev/ttyO4", 115200)
#   Serial using the USB port
#skywire = serial.Serial("/dev/ttyUSB2", 115200)

#   Declare latitude and longitude
latitude = 0
longitude = 0

"""
Desc:   Function that sends AT Command command
        and waits for expected response response back
        from the modem.
Input:  command, response
Output: Prints UI to termianl to let user know what is
        going on
"""
def sendCommand(command, response):
    global skywire
    line = ""
	#	Send the command to the Skywire
    skywire.write(command)
	#	Wait for expected response.
    while response not in line:
		#	If you get ERROR, try sending the command again
        if "ERROR" in line:
            print("Resending command...\r\n")
            skywire.write(command)
        line = skywire.readline()
    print(line + "\r\n")
    return

"""
Desc:   Function that reads from the modem via serial and
        continues to read until it gets the expected
        response str
Input:  str
Output: Prints UI to termianl to let user know what is
        going on
"""
def waitForResponse(str):
    global skywire
    line = ""
    while (str not in line):
        line = skywire.readline()
    print(line + "\r\n")
    return

"""
Desc:   Sets up GPS on the DE910/LE910 to get GPS information from
        it.
Input:  none
Output: Prints UI to termianl to let user know what is
        going on
"""
def setupGPS():
    global skywire
    print("Powering down GPS stuff...")
    sendCommand("AT$GPSP=0\r\n", "OK")
    if DE910:
        print("Getting GPS Lock...")
        sendCommand("AT$GPSLOCK=0\r\n","OK")
        print("Getting location...\r\n")
        sendCommand("AT$LOCATION=1\r\n", "OK")

    print("AT$GPSAT...\r\n")
    sendCommand("AT$GPSAT=1\r\n", "OK")

    print("AT$GPSP...\r\n")
    if DE910:
        sendCommand("AT$GPSP=1\r\n", "#GPS_STATUS:1")
    else:
        sendCommand("AT$GPSP=0\r\n", "OK")
        sendCommand("AT$GPSP=1\r\n", "OK")
    #else:
    #    print("Error, quitting...")
#        break

    if DE910:
        print("AT$NMEA...\r\n")
        sendCommand("AT$NMEA=1\r\n", "OK")
		
    return

"""
Desc:   Gets GPS signal and updates the latitude and longitude variables
Global: Updates global variables latitude and longitude
Input:  none
Output: Prints UI to termianl to let user know what is
        going on
"""
def getGPS():
    global skywire, latitude, longitude
    gps = ""
    gps_split = ""

    #   Get GPS from Skywire
    line = ""
    line_split = ""
    while True:
        print("Reading line...\r\n")
        skywire.write("AT$GPSACP\r\n")
        line = skywire.readline()
        if "$GPSACP" in line:
            line.split()
            print(line)
            line_split = line.split(",")
            print(line_split[0])
            if "00000" not in line_split[0]:
                latitude = line_split[1]
                longitude = line_split[2]
                print(line_split[0])
                waitForResponse("OK")
                break
        sleep(.1)
    
	#	Get latitude
	#   Modem doesn't return a negative N degree,
    #   instead returns a degree S. Google Maps needs
    #   degrees to be in N and E. So, if latitude is in
    #   S, make negative
    #   NOTE: THIS IS TAKING RAW DEGREES AND DOESN'T
    #   INCORPORATE MINUTES. NOT ACCURATE.
	print("Getting latitude...")
    if 'S' in latitude:
        latitude = -(float(latitude[:-1]) / 100)
    else:
        latitude = float(latitude[:-1]) / 100

    #	Get longitude
	#   Modem doesn't return a negative E degree,
    #   instead returns a degree W. Google Maps needs
    #   degrees to be in N and E. So, if latitude is in
    #   W, make negative
    #   NOTE: THIS IS TAKING RAW DEGREES AND DOESN'T
    #   INCORPORATE MINUTES. NOT ACCURATE.
    print("Getting longitude...")
    if 'W' in longitude:
        longitude = -(float(longitude[:-1]) / 100)
    else:
        longitude = float(latitude[:-1]) / 100

    print("Latitude = " + str(latitude) + " Longitude = " + str(longitude) + "\r\n")
    return

"""
Desc:   Gets MEID of the curret DE910/IMEI of the LE910
Global: Updates global variable URL
Input:  none
Output: Prints UI to terminal to let user know what is
        going on
"""
def getMEID():
    global skywire

    #   Setup line to be read to
    line = ""
    #   Setup line splits
    line_split1 = ""
    line_split2 = ""
    skywire.write("AT#MEIDESN?\r\n")
    while "#MEIDESN:" not in line:
        if "ERROR" in line:
            print("Resending command...\r\n")
            skywire.write("AT#MEIDESN?\r\n")
        line = skywire.readline()
    #   Split line to parse out '#MEIDESN: '
    line_split1 = line.split(": ")
    #   Split line to parse out MEID
    line_split2 = line_split1[1].split(",")
    return line_split2[0]

"""
Desc:   Gets IMEI of the curret LE910
Global: Updates global variables DEVICE_ID and URL
Input:  none
Output: Prints UI to terminal to let user know what is
        going on
"""
def getIMEI():
    global skywire
    #   Setup line to be read to
    line = ""
    skywire.write("AT+GSN\r\n")
    while True:
        if "ERROR" in line:
            print("Resending command...\r\n")
            skywire.write("AT+GSN\r\n")
        line = skywire.readline()
        #   >= 15 means we've gotten our IMEI
        #   Should be 17 exactly, but this way we
        #   account for white space chars not being
        #   picked up
        if len(line) >= 15:
            #   Strip \r\n off of the IMEI: it breaks requests...
            line = line[:-2]
            break
    return line

"""
Desc:   main funciton that runs the program. Sends the GPS
        information about every five seconds.
"""
def main():
    global skywire, latitude, longitude
    url = ""
    device_id = ""
    #   Soft reset of modem, just in case...
    print("Resetting modem...")
    sendCommand("ATZ\r\n", "OK")

    #   Turn echo off
    #   Makes reading from Skywire much easier!
    print("Turning echo off...\r\n")
    sendCommand("ATE0\r\n", "OK")

    if DE910:
        print("Getting MEID of modem...")
        device_id = getMEID()
    elif LE910:
        print("Getting IMEI of modem...")
        device_id = getIMEI()
    else:
        print("Modem not recognized. Make sure to put 'True' next to the modem")
        print(" you are using, and 'False' next to the modem you aren't using.\r\n")
        print("Quitting program.")
        sys.exit(0)

	#	Get rid of excess whitespace
    device_id.strip()

    #   Setup the URL for dweet
    url = "http://dweet.io/dweet/for/" + device_id
    url.strip()
    print("Your device ID is: " + device_id + "\r\n")
    print("Your URL is: " + url + "\r\n")

    #   Setup GPS
    setupGPS()

    while True:
        print("Getting GPS\r\n")
        getGPS()
        data = {'Latitude': latitude, 'Longitude': longitude}
        # Send response
        print("Sending information...\n")
        print url
        print data
        send_response = requests.post(url, data)
        print(send_response.url)
        print ("\n")
        print(send_response.text)
        print("\n\n")
        sleep(5)

if __name__== '__main__':
    main()
